"use client";

import { use, useEffect, useState } from "react";
import { notFound } from "next/navigation";
import wordpressAPI from "@/lib/wordpress";
import ContentCard from "@/components/cards/ContentCard";
import Pagination from "@/components/common/Pagination";
import SectionHeader from "@/components/common/SectionHeader";
import { useLanguage } from "@/contexts/LanguageContext";

interface CategoryProps {
  params: Promise<{ slug: string }>;
}

export default function CategoryArchive({ params }: CategoryProps) {
  const { slug } = use(params);
  const { language } = useLanguage();

  // State management
  const [posts, setPosts] = useState<any[]>([]);
  const [category, setCategory] = useState<any | null>(null);
  const [loading, setLoading] = useState(true);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);

  // Posts per page limit
  const LIMIT = parseInt(process.env.NEXT_PUBLIC_POSTS_LIMIT || "6", 10);

  // Fetch category info and posts
  useEffect(() => {
    const fetchCategoryData = async () => {
      try {
        setLoading(true);

        // Fetch category info
        const categoryInfo = await wordpressAPI.getCategoryBySlug(slug, language);
        if (!categoryInfo) {
          notFound();
          return;
        }
        setCategory(categoryInfo);

        // Fetch posts by category
        const data = await wordpressAPI.getPostsByCategory(slug, language, LIMIT, currentPage);
        setPosts(data.items || []);
        setTotalPages(data.totalPages || 1);
      } catch (error) {
        console.error("Error loading category data:", error);
        setPosts([]);
        setTotalPages(1);
      } finally {
        setLoading(false);
      }
    };

    fetchCategoryData();
  }, [slug, language, currentPage]);

  if (loading && !category) {
    return (
      <div className="te-category-archive my-12">
        <div className="container">
          <div className="animate-pulse">
            <div className="h-8 bg-gray-200 dark:bg-gray-700 rounded w-1/3 mb-4" />
            <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-2/3 mb-8" />
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="te-category-archive my-12">
      <div className="container">

        <div className="mb-8">
          <div className="grid grid-cols-[1fr_auto] gap-6 items-end">
            <div>
              {/* Category Header */}
              <SectionHeader
                title={category?.name || slug}
                subtitle={category?.description || `Posts in ${category?.name || slug} category`}
                level="h2"
                marginBottom="0"
                textAlign="start"
              />
            </div>

            <div className="w-auto">
              {/* Category Info */}
              {category?.count && (
                <div className="text-sm text-gray-600 dark:text-gray-400">
                  {language === "ar"
                    ? `${category.count} منشور في هذا التصنيف`
                    : `${category.count} Posts`}
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Posts Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {loading
            ? Array.from({ length: LIMIT }).map((_, i) => (
              <div
                key={i}
                className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden"
              >
                <div className="aspect-[3/2] w-full bg-gray-200 dark:bg-gray-700" />
                <div className="p-6 space-y-4">
                  <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-5/6" />
                </div>
              </div>
            ))
            : posts.length > 0
              ? posts.map((post) => (
                <ContentCard key={post.id} post={post} basePath="posts" />
              ))
              : (
                <div className="col-span-full text-center py-12">
                  <p className="text-gray-500 dark:text-gray-400">
                    {language === "ar"
                      ? "لا توجد منشورات في هذا التصنيف"
                      : "No posts found in this category"
                    }
                  </p>
                </div>
              )}
        </div>

        {/* Pagination */}
        {totalPages > 1 && (
          <Pagination
            currentPage={currentPage}
            totalPages={totalPages}
            onPageChange={(page) => setCurrentPage(page)}
          />
        )}
      </div>
    </div>
  );
}